/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * \file    DM388_i2c.h
 *
 * \brief   Defines the module register defines for I2C
 *
 * This header file contains the Macros definitions and the
 * register overlay structure for DM388 I2C controller.
 *
 * \author  0043
 *
 * \version 01a,13aug,2010 Created.
 */

#ifndef _DM388_aic3106_H_
#define _DM388_aic3106_H_

#ifdef __cplusplus
 extern "C" {
#endif


/*
 *====================
 * Includes
 *====================
 */
#include "dm388_types.h"

/**
@addtogroup DM388 AIC3106 Audio Codec
@{
*/

/*
 *====================
 * Defines
 *====================
*/
#define AIC3104_I2C_ADDR          0x18    // I2C address
/**< I2C Slave address of the AIC3106 Audio Codec chip */

#define AIC3104_I2C_BUS_NUM		2
/**< Bus Number where AIC3106 is connected on the EVM */

#define AIC3104_REG_ADDR_MASK	0x007F;
/**< Mask for AIC316 internal register offsets */

#define 	DM88_AIC3104_MCASP	0x4803C000u

/**< Base address of the McASP being used for AIC3106 on Base Board */

#define AIC3104_PAGESELECT        0
/**< AIC3106 Page Select */

#define AIC3104_RESET             1
/**< AIC3106 reset */
/*
 *====================
 * Function declarations
 *====================
 */
/** **************************************************************************
 * \n \brief Routine to set the register value in the AIC3106 Codec
 *
 * This routine initializes the I2C instance 0. We are not configuring the 
 * interrupts here as the code is intended to carry out all I/O operations
 * in polled mode.
 *
 * \param u8RegNum		[IN]	Bus Instance number
 * \param u8RegVal		[IN]	Value to be written
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
extern INT32 aic3104RegSet
(
	UINT8	u8RegNum,
	UINT8  	u8RegVal
);

/** **************************************************************************
 * \n \brief Routine to configure the Headphone Out and play sine tone
 *
 * This routine initializes the AIC3106 Audio Codec on the base board and
 * plays a sine tone on the head-phone.
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
INT32 aic3104_headphone_out
(
	UINT8    u8McaspNum
);

/** **************************************************************************
 * \n \brief Routine to configure the Headphone Out and play sine tone
 *
 * This routine initializes the AIC3106 Audio Codec on the base board and
 * plays a sine tone on the head-phone.
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
INT32 aic3104_lineout_out
(
	UINT8    u8McaspNum
);

/** **************************************************************************
 * \n \brief Routine to configure AIC3106 to loop Line-In to Line-Out
 *
 * This routine initializes the AIC3106 Audio Codec on the base board to take
 * input from the Line-In and play it back to Line-Out.
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
INT32 aic3104_LineInLineOut_Loop
(
	UINT8    u8McaspNum
);

/** **************************************************************************
 * \n \brief Routine to configure AIC3106 to loop Mic to head-phone
 *
 * This routine initializes the AIC3106 Audio Codec on the base board to take
 * input from the microphone and play it back on the headphones.
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
INT32 aic3104_MicHeadphone_Loop
(
	UINT8    u8McaspNum
);

/** **************************************************************************
 * \n \brief Routine to test the AIC3106 Audio codec on Base board
 *
 * This routine initializes the AIC3106 Audio Codec on the base board to test
 * various interfaces including the following
 *	(a) Playback on Head-phohne
 *  (b) Playback on Line-Out
 *  (c) Loopback the input on Microphone to head-set.
 *  (d) Loopback the input on Line-In to Line-Out
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
INT32 aic3104_tests
(
	UINT8    u8McaspNum
);


/* @} */   /* End of AIC3106 Audio Codec */

#ifdef __cplusplus
} /* End of extern C */
#endif /* #ifdef __cplusplus */

#endif /* _DM388_aic3104_H_ */
